#!/usr/bin/perl -s
#
# Syntax: sub2latex.pl <file.xml >file.tex
#         to read ``new'' dives (see below $stampDIR and $stampFILE).
#
# Converts an XML file produced by subsurface into a LaTeX file to be
# processed with logbook.cls. It is possible to select only newer dives
# using a "stampfile" (see below).
#
# Copyright © 2011-2019 by Daniel Flipo <daniel.flipo@free.fr>
# Version: 3.3b (2019/08/14)
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

use POSIX;
use strict;
use warnings;

#if ($#ARGV != 2)
#   {die "Usage: sub2latex.pl <file.xml >file.tex\n";}

# $ENV{HOME} is usually uninitialized on Windows :-(
if ($^O eq "MSWin32" && $ENV{HOME} eq "")
   {$ENV{HOME} = "C:"}

# User customisable variables:
# You can change the values of the next four variables if you wish.
# Dive is completed when depth remains under $mindepth (meters):
my $mindepth=0.5;
# GF values will be copied to the .tex file only if at least one
# of them is >= $GFmin (%).
my $GFmin=50;
# Default dir/file for recording last dive stamp (user changeable variables),
my $stampDIR="$ENV{HOME}/.divetools";
my $stampFILE=".lastdive-sub";
# ASCENT alarm will be added if ascending speed exceeds this limit (in m/min.).
my $ASClim=11;
# Depth precision in profiles (default= 2 decimal digits = 1cm).
# OSTC 2N provides 3 decimal digits, Vyper 2, Nemo2 just 1.
# This will be reduced to $prec="%.1f" for Mares Nemo Wide computers.
my $prec="%.2f";
#  Change this only for debugging:
my $UpdateStamp=1; # =1 to read only new dives, =0 for debugging

# Please leave the rest unchanged.

# Global variables for this script:
my $CCRmode=0;
my $DCunknown=1;
my $Desat="";
my $DiluentSet=0;
my $GF=0;
my $GFhi=0;
my $GFlo=0;
my $LastDive=0;
my $OTUday=0;
my $PrintDiveNR=1;
my $SPevent=0;
my $Surf="";
my $airpressure=1000; # mbars
my $airtemp=0;
my $altitude=0;
my $avgdepth=0;
my $buddy="";
my $ceil=0;
my $cns=0;
my $cnsmax=0;
my $decomode="";
my $delta=0;
my $deltamode=0;
my $density=0;
my $deviceid="";
my $diveNR=0;
my $dives=0;
my $dvarea="";
my $dvdate="";
my $dvduration="";
my $dvlat="";
my $dvlong="";
my $dvspot="";
my $dvtime="";
my $firmware="";
my $gravity=9.81; # m/s^2
my $indexw=0;
my $laststop=3;   # m
my $lasttic=0;
my $maxTTS=0;
my $maxTTSdepth=0;
my $maxTTStic=0;
my $maxascent=0;
my $maxdepth=0;
my $mbarpermeter=100;
my $ndl=200;
my $notes="";
my $pdvdate="";
my $pend=0;
my $ppo2=0;
my $previousGAS="";
my $previousPO2=0;
my $pstart=0;
my $serial="";
my $stopdepth=0;
my $stoptime=0;
my $timeend=0;
my $watertempavg=0;
my $watertempmax=0;
my $watertempmin=0;
my $wtemp=0;
my %serialID=();
my @GASlist=(0);
my @GASswitch=(0);
my @MAXdeco=();
my @SPlist=(0);
my @alarm=(0);
my @ascent=(0);
my @ceiling=(0);
my @divesites=();
my @dprofile=();
my @gfprofile=(0);
my @mix=();
my @tankvol=();
my @tmp=(0);
my @tprofile=();

# Subroutine to convert the mixnames : uppercase, strip spaces and
# convert to a TeX control sequence if the name is language dependent.
sub localise_mixname {
    my $name = shift;
    $name = uc($name);
    $name =~ s/\s//g;
    $name =~ s/AIR/\{\\dlAIR\}/;
    $name =~ s/EAN21/\{\\dlAIR\}/;
    $name =~ s/LUFT/\{\\dlAIR\}/;
    $name =~ s/SAUERSTOFF/\{\\dlOXY\}/;
    $name =~ s/OXYGEN/\{\\dlOXY\}/;
    $name =~ s/EAN100/\{\\dlOXY\}/;
    # EAN99, 98, 97, 96, 95 are oxygen
    $name =~ s/EAN99/\{\\dlOXY\}/;
    $name =~ s/EAN98/\{\\dlOXY\}/;
    $name =~ s/EAN97/\{\\dlOXY\}/;
    $name =~ s/EAN96/\{\\dlOXY\}/;
    $name =~ s/EAN95/\{\\dlOXY\}/;
    $name =~ s/TMX/TX/;
    return $name;
}

# Subroutine to compute O2 percentage from the mix name (EAN32, TX20/30, etc.)
# as formated by localise_mixname.
sub O2_from_mixname {
    my $name = shift;
    my $o2 = 0;
    if ($name =~ /AIR/i)
       {$o2 = 21;}
    elsif ($name =~ /EAN(\d+)/i)
       {$o2 = $1;}
    elsif ($name =~ /TX(\d+)\//i)
       {$o2 = $1;}
    elsif ($name =~ /OXY/i)
       {$o2 = 100;}
    else
      {print STDERR "Unknown gas: $name, OTU not computed for $name!\n";}
    return $o2;
}

# Subroutine to strip extra braces in <gasname>: \Gas<n>{<gasname>}{}{}
sub strip_braces {
    my $name = shift;
    $name =~ s/^\{//;
    $name =~ s/\}$//;
    return $name;
}

# Subroutine to add a new gas to @GASlist (checking that it is not already in)
sub add_new_gas {
    my $name = shift;
    my $flag = 1;
    foreach (@GASlist) {
       $flag = 0 if ($_ eq $name);
    }
    push (@GASlist, $name) if $flag;
}

# Subroutine to add a new setpoint to @GASlist (if not already in)
sub add_new_setpoint {
    my $name = shift;
    my $flag = 1;
    foreach (@SPlist) {
       $flag = 0 if ($_ eq $name);
    }
    push (@SPlist, $name) if $flag;
}

# Subroutine to compute « cns clock » from ppo2 given in bar. Based on
# « NOAA CNS Percentage Exposure Table », checked with cns.pl script.
sub cns_clock {
    my $p = shift;  # bar
    my $cns=0;
    my $a=27; my $A=7000; my $B=6.5; my $C=2.2;
    if ($p > 0.6 && $p <= 1.5)
       {$cns = 1 / (5.7 - 3 * $p);}
    elsif ($p > 1.5)
       {$cns = 5/6 + ($a *($p - 1.5))**6 / $A + ($a *($p - 1.5))**2 / $B
                       +  $C *($p - 1.5);
       }
    return $cns;
}

# Given a divesite id, find its name, gps coordinates… in array @divesites
sub divesite_elements {
    my $id = shift;
    my $k = 0;
    my $dvgps = "";
    while ($divesites[$k] ne $id && $k < $#divesites)
          {$k+=3;}
    if ($k < $#divesites)
       {$dvspot= $divesites[$k+1];
        $dvgps = $divesites[$k+2];
        if ($dvgps =~ /([-]?[\d]+.[\d]+)(\s)([-]?[\d]+.[\d]+)/i)
           {my $lat  = $1;
            my $long = $3;
            if ($lat < 0)
               {$dvlat = decdeg_to_deg_min_sec(abs($lat))   . "\\dlSouth";}
            else
               {$dvlat = decdeg_to_deg_min_sec($lat)        . "\\dlNorth";}
            if ($long < 0)
               {$dvlong = decdeg_to_deg_min_sec(abs($long)) . "\\dlWest";}
            else
               {$dvlong = decdeg_to_deg_min_sec($long)      . "\\dlEast";}
           }
        elsif ($dvgps ne "")
           {die("Wrong format for GPS field: $!");}
       }
}

# Convert decimal degrees (GPS coordinates as stored in subsurface XML file)
# into degres, minutes, seconds (with 1 decimal only).
# WARNING: the argument of this subroutine is supposed to be NON NEGATIVE.
sub decdeg_to_deg_min_sec {
    my $decdeg = shift;
    my $deg = POSIX::floor($decdeg);
    my $min = POSIX::floor(60 * ($decdeg - $deg));
    my $sec = 3600 * (($decdeg - $deg) - $min/60);
    # %04.1f : 1 decimal, 4 characters min. including the `.'
    my $dms = sprintf("%02d°%02d'%04.1f\"", $deg, $min, $sec);
    return $dms;
}

# Unless $UpdateStamp==0, read the stampfile "$stampDIR/$stampFILE"
# to find out the last dive already logged in LaTeX: this number will be
# updated automatically at the end of this script.
unless ($UpdateStamp ==0 )
  {if (open(my $in, "<", "$stampDIR/$stampFILE"))
      {while (my $line = <$in>)
             {if ($line =~ /LastDive=(\d\d*)/i) # 1 or more digits
                 {$LastDive=$1;}
              else
                 {print(STDERR "'LastDive=' not found, all dives logged!\n");}
             }
       close($in);
      }
   else
      {print(STDERR "Can't open $stampDIR/$stampFILE, all dives logged!\n");
      }
  }

# Preamble: custom (read from file "logbook.pre") or standard
if (open(my $in, "<", "logbook.pre"))
 {while (<$in>) { print("$_");}
  close $in;
 }
else
 {print("\\documentclass[sorted,french]{logbook}\n\n");
  print("\\usepackage[utf8]{inputenc}\n");
  print("\\usepackage{fourier}\n");
  print("\\pagestyle{empty}\n");
 }

# \begin{document}
print("\n\\begin{document}\n");

while (<>)
  {# <settings> ... </settings>
   # libdivecomputer produces weird deviceid, subsurface provides serial number
   # (also called nickname) in <settings> ... </settings> (see tag
   # <divecomputerid >).  We store them in a "hash" %serialID.
   if (/<divecomputerid\s+(.*)/i)
      {my $var=$1;
       if ($var =~ /deviceid='([^']+)/i)
          {$deviceid="$1";
           if ($var =~ /serial='([^']+)/i)
              {$serial="$1";
               %serialID =(%serialID, $deviceid, $serial);
              }
           elsif ($var =~ /nickname='([^']+)/i)
              {$serial="$1";
               %serialID =(%serialID, $deviceid, $serial);
              }
          }
       if ($var =~ /firmware='([^']+)/i)
          {$firmware=$1;}
      }
   # <divesites> ... </divesites> (version 3)
   if (/<site\s+(.*)/i)
      {my $var=$1; my $uuid=''; my $name=''; my $gps='';
       if ($var =~ /uuid='([^']*)/i)
          {$uuid=$1;}
       if ($var =~ /name='([^']*)/i)
          {$name=$1;}
       if ($var =~ /gps='([^']*)/i)
          {$gps=$1;}
       if ($uuid ne '' && $name ne '')
          {push (@divesites, ($uuid, $name, $gps));}
      }

   # Trips
   if (/<trip\s+.*\slocation='(.*)'\s*>/)
      {$dvarea=$1;}
   if (/<\/trip>/)
      {$dvarea="";}

   # DIVE loop
   if (/<dive\s+(.*)/i)
      {# read the attributes of the <dive ...> tag
       my $var=$1;
       if ($var =~ /number='(\d*)/i)
          {$diveNR=$1;}
       # Skip already recorded dives
       if ($diveNR > $LastDive) # If true, this is a new dive, read on.
          {print("\n\\begin{dive}\n\n");
           if ($PrintDiveNR)
              {print("\\DiveNr{$diveNR}\n");
               # Comment out next line to print the dive number
               #$PrintDiveNR=0;       # only for the first dive
              }
           else
              {print("\\DiveNr{}\n");}
           # Reset all variables (local to each dive)
	   $DCunknown=1;
	   $DiluentSet=0;
	   $GF=0;
	   $decomode="";
	   $lasttic=0;
	   $maxascent=0;
           $CCRmode=0;
           $Desat="";
           $GFhi=0;
           $GFlo=0;
           $SPevent=0;
           $Surf="";
           $airtemp=0;
           $altitude=0;
           $avgdepth=0;
           $ceil=0;
           $cns=0;
           $cnsmax=0;
           $delta=0;
           $deltamode=0;
           $density=0;
           $dives+=1;
           $dvdate="";
           $dvduration="";
           $dvlat="";
           $dvlong="";
           $dvspot="";
           $dvtime="";
           $indexw=0;
           $maxTTS=0;
           $maxTTSdepth=0;
           $maxTTStic=0;
           $maxdepth=0;
           $mbarpermeter=100;
           $ndl=200;
           $notes="";
           $pend=0;
           $ppo2=0;
           $previousGAS="";
           $previousPO2=0;
           $pstart=0;
           $stopdepth=0;
           $stoptime=0;
           $watertempavg=0;
           $watertempmax=0;
           $watertempmin=0;
           $wtemp=0;
           @GASlist=(0);
           @GASswitch=(0);
           @MAXdeco=();
           @SPlist=(0);
           @alarm=(0);
           @ascent=(0);
           @ceiling=(0);
           @dprofile=();
           @gfprofile=(0);
           @mix=();
           @tankvol=();
           @tmp=(0);
           @tprofile=();
           if ($var =~ /divesiteid='([^']*)/i)
              {divesite_elements($1);}
           if ($var =~ /date='([^']*)/i)       # (2011-12-10')
              {$dvdate=$1;}
           if ($var =~ /time='([^']*)/i)       # (10:21:00')
              {$dvtime=$1;}
           if ($var =~ /duration='(\d*:\d*)/i) # (65:50 m')
              {$dvduration=$1;}
           if ($dvdate =~ /(\d*)(-)(\d*)(-)(\d*)/)
              {print("\\DiveDate{$1}{$3}{$5}\n");}
           if ($dvtime =~ /(\d*)(:)(\d*)/)
              {print("\\TimeStart{$1}{$3}\n"); # seconds dropped...
               $dvtime = $1 * 60 + $3;         # NOW $dvtime is in minutes
              }                                # used later to compute $timeend
           if ($dvduration =~ /(\d*)(:)(\d*)/)
              {print("%\\Duration{$1}{$3}\n");} # for comparison, see below
          }
       else
          {$diveNR=0;}
      } # closing if (/<dive (.*)/i) { ...

# Skip already recorded dives
if ($diveNR > $LastDive)  # If true, this is a new dive, record all tags
   {# Pick up user provided informations (if any) for future printing:
   if (/<location\s*[^>]*>(.*)<\/location>/i)
      {$dvspot=$1;}
   if (/<buddy>(.*)<\/buddy>/i)
      {$buddy=$1;}

   # <notes> ... </notes>  (single line)
   if (/<notes>(.*)<\/notes>/i)
      {$notes = "$1" . "\n";}
   # <notes> ... </notes>  (multiline)
   elsif (/<notes>(.*)/i)
      {$notes = $1;
       local $/ = "</notes>\n";     # locally redefine the "line" separator
       local $_ = <>;               # read next "line"
       s/<\/notes>//;               # strip </notes> tag from "line" ($_)
       $notes= $notes . "\n" . $_ ;
      }

   # New cylinder?
   if (/<cylinder(.*)/i)
      {my $var=$1;
       my $o2=21; my $he=0; my $pstart=""; my $pend=""; my $mixname="";
       # Tank volume (liters)
       if ($var =~ / size='([\d\.]*)/i)
          {push (@tankvol, $1);}
       else
          {push (@tankvol, "");}
       # pstart, pend
       if ($var =~ / start='([\d\.]*)/i) # (205.0 bar')
          {$pstart=$1;}
       if ($var =~ / end='([\d\.]*)/i)   # (65.0 bar')
          {$pend=$1;}
       # Mixname
       if ($var =~ / o2='(\d*)/i)   # NO decimal part for $mixname!
          {$o2=$1;}
       if ($var =~ / he='(\d*)/i)   # NO decimal part for $mixname!
          {$he=$1;}
       if ($he == 0)
          {$mixname = "EAN" . "$o2";}
       else
          {$mixname = "TX" . "$o2" . "/" . "$he";}
       push (@mix, (localise_mixname($mixname), $pstart, $pend, 0));
       # use= ???
       #if ($var =~ / use='(\d)/i)  # use='1' for CC gases?
       #   {$use=$1;}
      }

   # Divecomputer details
   if (/<divecomputer(.*)/i)
      {my $var=$1;
       $DCunknown=0;
       # Print site and buddy information (once!)
       print("\\Area{$dvarea}\n");
       print("\\Spot{$dvspot}\n");
       print("\\GPS{$dvlat}{$dvlong}\n") if ($dvlat ne '');
       print("\\Buddy{$buddy}\n\n");
       # Computer specifications
       if ($var =~ / model='([^']*)/i)
          {my $model=$1;
           # Mares Nemo Wide only provides 1 digit when recording depths
           $prec="%.1f" if ($model =~ /Nemo/);
           if ($model =~ /(Heinrichs Weikamp)\s+(.*)/i)
              {print("\\Vendor{$1}\n");
               print("\\Model{$2}\n");
              }
           elsif ($model =~ /OSTC\s+/i)
              {print("\\Vendor{Heinrichs Weikamp}\n");
               print("\\Model{$model}\n");
              }
           elsif ($model =~ /([^ ]*)\s(.*)/i)
              {print("\\Vendor{$1}\n");
               # Correction for "Nemo Wide 2" (too long name) -> "Nemo 2"
               my $zz=$2; $zz =~ s/Wide\s//;
               print("\\Model{$zz}\n");
              }
           if ($model =~ /Mares/i || $model =~ /Suunto/i|| $model =~ /Cressi/i)
              {$decomode="RGBM";}
           elsif ($model =~ /OSTC/i)
              {$decomode="L16-GF";}
           else
              {$decomode="Bühlmann";}
          }
       # Serial number
       if ($var =~ / serial='([^']*)/i)
          {$serial = "$1";}
       elsif ($var =~ / deviceid='([^']*)/i)
          {$deviceid = "$1";
           if (%serialID ne "0" && $serialID{$deviceid} ne "")
              {$serial="$serialID{$deviceid}";}
           }
       # Firmware version number
       if ($var =~ / firmware='([^']*)/i)
          {$firmware = "$1";}
       # dctype attribute
       if ($var =~ /dctype='([^']*)/i)
          {$CCRmode = 1 if ($1 eq "CCR");
          }

       # Surface interval (will be printed later on)
       if ($var =~ / surfacetime='(\d*)/i) # minutes
          {$var=$1;
           if ($var > 60)
              {$Surf=sprintf("\\SurfInt{%u}{%02s}", int($var/60), $var%60);}
           else
              {$Surf=sprintf("\\SurfInt{}{%u}", $var);}
           if ($dvdate ne $pdvdate)
              {$OTUday = 0;
               $cns = 0;
              }
           else
              {$cns = $cns / 2**($var/90);}
           $pdvdate = $dvdate;
          }
       else
          {# Compute surface interval for dives in chronological order
           # AND performed on the same day.
           if ($dvdate eq $pdvdate && $dvtime > $timeend)
              {$var=$dvtime - $timeend;
               if ($var > 60)
                  {$Surf=sprintf("\\SurfInt{%u}{%02s}", int($var/60), $var%60);}
               else
                  {$Surf=sprintf("\\SurfInt{}{%u}", $var);}
               $cns = $cns / 2**($var/90);
              }
           else   # 1rst dive of the day
              {$Surf=sprintf("\\SurfInt{\\textemdash}{}");
               $OTUday = 0;
               $cns = 0;
              }
           $pdvdate = $dvdate;
          }
      }

    # Depth max/mean
    if (/<depth(.*)/i)
       {my $var=$1;
        # Warning in case of unknown dive computer
        unless ($DCunknown == 0)
           {print STDERR "\nWARNING: <divecomputer /> tag not found\n";
            print STDERR "The .tex file might be incomplete. Input line $.\n";
           }
        if ($var =~ / max='([\d\.]+)/i)
           {$maxdepth = $1;}
        if ($var =~ / mean='([\d\.]+)/i)
           {$avgdepth = $1;}
       }

    # TEMPERATURE means "minimal water temperature"
    if (/<temperature(.*)/i)
       {my $var=$1;
       if ($var =~ / air='([\d\.]+)/i)   # ('23.1 C')
          {$airtemp=$1;}                    # not used in logbook.cls
       if ($var =~ / water='([\d\.]+)/i) # ('23.1 C')
          {$watertempmin=$1;
           $wtemp=$1;          # initial value for $watertempavg (not 0!)
          }
       }

    # Salinity in g/l (not saved by subsurface?)
    if (/<water salinity='([\d\.]+)/i)
       {$density = $1 * 1000 if ($1 < 2);
        $density = $1        if ($1 >950);
        $mbarpermeter = $density * $gravity / 100;
       }

    # Ambient air pressure in mbar
    if (/<surface pressure='([\d\.]+)/i)
       {$airpressure = $1 * 1000;}

    # Extra data (OSTC specific?)
    if (/<extradata(.*)/i)
       {my $var=$1;
        if ($var =~ / key='Serial'\s+value='([^']+)/i)
           {$serial=$1;}    # overides serial set in <divecomputerid />
        if ($var =~ / key='FW Version'\s+value='([^']+)/i)
           {$firmware=$1;}  # overides firmware set in <divecomputerid />
        if ($var =~ / key='Desat time'\s+value='(\d+):(\d+)/i)
           {$Desat="\\Desat{$1}{$2} % h:mn\n";}
        if ($var =~ / key='Deco model'\s+value='([^']+)/i)
           {$decomode = $1;
            $decomode =~ s/^ZH-//;
            $CCRmode=1 if ($decomode =~ /\sCC$/);
            $decomode =~ s/\s[OC]C$//;
           }
        if ($var =~ / key='Deco model info'\s+value='GF\s+(\d+)\/(\d+)/i)
           {$GF = 1;  # Delay GF printing, see below in tag <event>.
            $GFlo = $1; $GFhi = $2;
           }
       }

    # Events
    if (/<event(.*)/i)
       {my $var=$1; my $t=0; my $o2=21; my $he=0; my $mixname="";
        # Print Divecomputer infos, DecoMode and GF now, once only!
        if ($decomode ne "")
           {print("\\Serialid{$serial}\n")  if ($serial ne "");
            print("\\Version{$firmware}\n") if ($firmware ne "");
            if ($CCRmode == 1)
               {$decomode= "$decomode" . " CC";}
            elsif ($decomode ne "RGBM" && $decomode ne "Bühlmann")
               {$decomode= "$decomode" . " OC";}
            print("\\DecoMode{$decomode}\n");
            if ($GF == 1)
               {print("\\GFlo{$GFlo} \\GFhi{$GFhi}\n");}
            elsif ($decomode =~ /GF/i) # default values for OSTC
               {print("\\GFlo{30} \\GFhi{90}\n");}
            $decomode = "";
            print("\\CCRmodetrue\n") if ($CCRmode == 1);
            #
            print("\\Density{$density} % g/l\n")          if ($density != 0);
            print("\\AirPressure{$airpressure} % mbar\n") if ($airpressure != 1000);
            # Now print Surface interval (and Desat if available)
            print("$Surf\n");
            print("$Desat") if ($Desat ne "");
            print("\n");
           }
        # Log time in seconds
        if ($var =~ / time='(\d+):(\d+)/i)
           {$t = $1*60+$2;}
        # Gas changes are logged in <event ... /> tags: name='gaschange'
        # Bailout switches should be tagged differently…
        if ($var =~ / name='gaschange'/i)
           {if ($var =~ / cylinder='(\d*)/i)
               {my $gasindex = $1;               # cylinder index
                $mixname=$mix[($gasindex-1)*4]   # if 1rst cylinder index = 1
               }
            elsif ($var =~ / he='(\d*)/i)      # integer, no %
               {$he=$1;
                if ($var =~ / o2='(\d*)/i)     # integer, no %
                   {$o2=$1;}
                $mixname="TX" . $o2 . "/" . $he;
               }
            elsif ($var =~ / o2='(\d*)/i)
               {$o2=$1;
                $mixname="EAN" . $o2;
                $mixname=localise_mixname($mixname);
               }
            elsif ($var =~ / value='(\d*)/i)
               {if ($1 <= 100)
                   {$o2=$1;
                    $mixname="EAN" . $o2;
                   }
                else       # value = %he * 2^16 + %o2
                   {$o2=$1 % 65536;
                    $he=int($1 / 65536);
                    $mixname="TX" . $o2 . "/" . $he;
                   }
                $mixname=localise_mixname($mixname);
               }
            else
               {$mixname="EAN21";
                $mixname=localise_mixname($mixname);
               }
            # Check if this mix has been declared in <cylinder >;
            # if so, its name appears in @mix.
            my $k = 0;
            while($k < $#mix && $mix[$k] ne $mixname )
                 {$k=$k+4;}
            if ($k > $#mix)  # this is a new gas, add it to @mix array
               {print STDERR "Dive $diveNR: adding $mixname to 'mix' array. Mismatch in cylinders?\n";
                push (@mix, ($mixname, "", "", 0));
               }
            # Only consider gas switches when the mix really changes
            if ($mixname ne $previousGAS)
               {$previousGAS = $mixname;
                # Add this mix to @GASlist array if not yet in
                add_new_gas ($mixname);
                # Add this mix to @GASswitch with time=$t and depth=0,
                # (time and depth will be updated later)
                # WARNING: in CCRmode, GAS switches appear first (in <event>),
                # sorted in chronological order, then SP switches are recorded
                # in <sample>, they appear later but in chronological order too,
                # this is fine as long as there is no "bailout" switch which
                # triggers both color and style change.  Resorting @GASswitch
                # will be requested in CCRmode with "bailout" switches.
                if ($CCRmode == 1)
                   {# Quick and dirty hack: first gas change = Diluent,
                    # other gas changes = bailout
                    # TODO: special bailout tag in <event > requested!
                    if ($DiluentSet == 0)
                       {push (@GASswitch, ("$mixname", $t, 0));
                        $DiluentSet = 1;
                       }
                    else
                       {push (@GASswitch, ("BO=$mixname", $t, 0));}
                   }
                else
                   {push (@GASswitch, ("$mixname", $t, 0));}
               }
           }
        # SetPoint changes (CCR) are recorded in <sample > see below.
        # This is only for imported dives (uddf, etc.)
        if ($var =~ / name='SP change'/i)
           {$SPevent = 1;
            if ($var =~ /value='([^']*)/i) # decimal OR integer!
               {my $sp = $1;
                $sp = $sp / 1000 if ($sp > 100); # $sp in mbar ?
                # Add this SetPoint to @SPlist array if not yet in
                add_new_setpoint($sp);
                # Add this SP to @GASswitch with time and depth (updated later)
                push (@GASswitch, ("SP=$sp", $t, 0));
               }
           }
        # Alarms are logged in <event ... /> tags under various names...
        elsif ($var =~ /name='(.*)(alarm|ascent|deco|ppo2|ceiling|stop|error|bookmark)([^']*)/i)
           {my $alm = uc("$1" . "$2" . "$3");
            $alm =~ s/\s/_/g;
            push (@alarm, ("$alm", $t, 0));
           }
       }

   # Samples
   if (/<sample(.*)/i)
      {my $var = $1; my $t=0; my $d=0;
       # Time
       if ($var =~ / time='(\d+):(\d+)/i)
          {$t = $1 * 60 + $2 ;
           # Time starts at 0 (like depth).
           push(@tprofile, 0)  if ($#tprofile < 0 && $t>0);
           push(@tprofile, $t);    # time tics in seconds
          }
       # Depth in meters. Keep full precision given by the computer to compute
       # ascending speeds. They will later be rounded to one or two decimal
       # digit for printing [Some computers, like Mares Nemo2, provide only
       # digit for others, like OSTC provide up to 3 digits!].
       if ($var =~ / depth='([\d\.]+)/i)
          {$d = $1;
           # First depth should always be 0.0!
           push(@dprofile, 0)  if ($#dprofile < 0 && $d>0);
           push(@dprofile, $d);
          }
       # Water temperature
       if ($var =~ / temp='([\d\.]+)/i) # means temp change!
          {$wtemp = $1;
           $indexw += 1;
           # Record maximum except during the first 3 minutes = 180 sec. and
           # in shallow waters (specially at end of dive)
           if ($t >= 180 && $d > $mindepth && $wtemp > $watertempmax)
              {$watertempmax = $wtemp;}
          }
       $watertempavg += $wtemp;           # no 'temp=' means no temp change

       # Deco / NDL / Ceiling
       if ($var =~ /tts='(\d+):(\d+)/i)
          {my $tts=$1*60+$2;              # seconds
           if ($maxTTS<$tts)
              {$maxTTS=$tts;
               $maxTTSdepth=$d;
               $maxTTStic=$#dprofile;
              }
          }
       # $ceil stores either 0 (no deco) or the former value of $stopdepth
       if ($var =~ /ndl='(\d+):(\d+)/i)
          {# Important: tags "ndl=0" have *no* effect on ceilings
           # (some of these tags are buggy!).
           $ndl=$1;      # $ndl in minutes ($ndl=$1*60+$2; # seconds)
           if ($ceil > 0 && $ndl > 0)       # deco finished
              {push (@ceiling, ($#dprofile, $ceil));
               $ceil=0;
               push (@ceiling, ($#dprofile, $ceil));
              }
          }
       if ($var =~ /stoptime='(\d+):(\d+)/i)
          {$stoptime=$1;} # stoptime in minutes (not used)
       if ($var =~ /stopdepth='([\d\.]*)/i)
          {$stopdepth=$1;
           if ($ceil != $stopdepth)         # change of deco depth
              {push (@ceiling, ($#dprofile, $ceil));
               $ceil=$stopdepth;
               push (@ceiling, ($#dprofile,$ceil));
              }
          }

       # CNS (only changes are printed), record only the max value.
       if ($var =~ / cns='(\d+)/i)
          {$cnsmax=$1 if ($1 > $cnsmax);}          # integer %
       # Setpoint changes (decimal numbers in bars), recorded in <sample>.
       # Add this Setpoint change to @SPlist and to @GASswitch
       if ($var =~ / po2='([\d\.]*)/i && $SPevent == 0)
          {if ($1 > 0)
              {my $po2=$1;                       # bar
               # Record po2 only when it really changes
               if ($previousPO2 != $po2)
                  {add_new_setpoint($po2);
                   push (@GASswitch, ("SP=$po2", $t, $d));
                   $previousPO2 = $po2;
                  }
              }
          }

       # GF profile (not supported by subsurface)
       #if ($var =~ /gf='(\d+)/i)
       #   {push(@gfprofile, $1);}
       # else
       #   {push(@gfprofile, 0);}

    } # end of <sample ... />

   if (/<\/dive>/i)
      {# In case @GASlist / @GASswitch are empty (no "gaschange" event),
       # fetch information from the <cylinder> tag if available.
       if (@GASlist < 2 && @mix > 0)
          {add_new_gas ($mix[0]);
           push (@GASswitch, ("$mix[0]", "0", "0.00"));
          }
       # Read the dprofile array backwards and remove the last values until
       # the first depth >= $mindepth, then add a final 0.0 value.
       my $i = $#dprofile;
       if ($#dprofile > 1)        # only if profile is not empty
          {while ($dprofile[$i] < $mindepth)
                {my $j = pop(@dprofile);
                 $i -=1;
                }
           push(@dprofile,"0.0") if ($dprofile[$#dprofile] != 0);
          }
       # *DO NOT* strip last values of @tprofile as $#tprofile is used
       # to compute $watertempavg!
       #$i = $#tprofile;
       #while ($tprofile[$i-1] > $#dprofile)
       #      {my $j = pop(@tprofile);
       #      $i -=1;
       #      }
       # # Strip last values of @ceiling accordingly (see HW test dive!):
       if ($#ceiling > 4)
          {my $i = $#ceiling;
           while ($ceiling[$i-1] > $#dprofile)
                 {my $j = pop(@ceiling);
                  $j = pop(@ceiling);
                  $i -=2;
                 }
          }
       # if the last point isn't at depth 0, close the graph
       if ($ceiling[$#ceiling] != 0)
          {push(@ceiling, ($#dprofile, $ceiling[$#ceiling]));
           push(@ceiling, ($#dprofile, 0));
          }
       # Strip last values of @gfprofile accordingly:
       #if ($GFdiv !=0 && $#gfprofile > 0)
       #   {my $k=$#gfprofile-int($#dprofile/$GFdiv);
       #    for ($i = 0; $i < $k; $i++)
       #        {my $j=pop(@gfprofile);}
       #   }
       #
       $GASlist[0] = $#GASlist;
       # Sort @SPlist by ascending numeric values:
       @SPlist = sort {$a <=> $b} @SPlist;
       $SPlist[0] = $#SPlist;
       $GASswitch[0] = $#GASswitch/3;
       $alarm [0] = $#alarm / 3;
       # Check whether time increments are constant or not:
       $i=$tprofile[1]; my $j=$i;
       if ($#tprofile > 1)
          {for (my $k=1 ; $k <= $#tprofile ; $k++)
               {my $var = $tprofile[$k] - $tprofile[$k-1] ; # time increments
                $i = $var if ($var > $i);                   # max increment
                $j = $var if ($var < $j && $var > 0);       # min increment
               }
           if ($i == $j)
              {$deltamode=1;
               $delta = $i;
               print("%DeltaMode\n");
              }
           else
              {$deltamode=0;
               $delta = 1;
               print("%Variable time increments\n");
              }
           }
       if ($#dprofile <= 1) # empty profile
          {$dvduration =~ s/:\d*$//;    # strip secondes
           print("\\Duration{$dvduration}\n");
           $timeend = $dvtime + $dvduration;
           printf("\\TimeEnd{%02s}{%02s}\n", int($timeend/60), $timeend%60);
           printf("\\WaterTempMin{%.1f}\n", $watertempmin);
           printf("\\MaxDepth{%.1f}\n",$maxdepth);
          }
       else
         {# real profile ($#dprofile > 1)
          # Events (alarms and gas switches) are recorded incompletely:
          # change time into tics and add the corresponding depth rounded
          # to 1 or 2 decimal digits.
          $i = 2;
          while ($i < @alarm)
                {$alarm[$i] = $alarm[$i] / $delta;
                 if ($deltamode == 1)
                    {$alarm[$i+1] = sprintf("$prec", $dprofile[$alarm[$i]]);}
                 else
                    {# interpolation needed to get the corresponding depth
                     my $j=0;
                     until($alarm[$i] <= $tprofile[$j])
                          {$j+=1;}
                     my $d = $dprofile[$j] + ($dprofile[$j-1]-$dprofile[$j])*($tprofile[$j]-$alarm[$i])/($tprofile[$j]-$tprofile[$j-1]);
                    $alarm[$i+1] = sprintf("$prec", $d);
                    }
                 $i+=3;
                }
          $i = 2;
          while ($i < @GASswitch)
                {$GASswitch[$i] = $GASswitch[$i] / $delta;
                 if ($deltamode == 1)
                    {$GASswitch[$i+1] = sprintf("$prec", $dprofile[$GASswitch[$i]]);}
                 else
                    {# interpolation needed to get the corresponding depth
                     my $j=0;
                     until($GASswitch[$i] <= $tprofile[$j])
                          {$j+=1;}
                     my $d = $dprofile[$j] + ($dprofile[$j-1]-$dprofile[$j])*($tprofile[$j]-$GASswitch[$i])/($tprofile[$j]-$tprofile[$j-1]);
                    $GASswitch[$i+1] = sprintf("$prec", $d);
                    }
                 $i+=3;
                }
          #
          # Compute ascent speeds from @dprofile increments (2 decimal digits)
          # and add an entry in @alarm in case it is higher than $ASClim
          # (only for fixed time increments).
          if ($deltamode == 1)
             {$i = 2; $j=1; my $asclim = $ASClim / 60 * $delta;
          # First build the @ascent array which holds the high ascending speeds:
             while ($i < @dprofile)
                   {my $asc = $dprofile[$i] - $dprofile[$i-1];
                    if ($asc <= -$asclim)
                       {$ascent[$j]   = sprintf("$prec", $asc * 60 / $delta);
                        $ascent[$j+1] = $i-1;
                        $ascent[$j+2] = sprintf("$prec", $dprofile[$i-1]);
                        $j+=3;
                    }
                    $i+=1;
                   }
             $ascent[0] = $#ascent / 3;
             # Compute maximal ascent speed
             if ($ascent[0] > 0)
                {for ($i = 1; $i < @ascent; $i+=3)
                   {$maxascent = $ascent[$i] if ($ascent[$i] < $maxascent);}
                 $maxascent = -$maxascent;
                }
          } # end of if ($deltamode == 1)
          # Merge alarms from @ascent and @alarm into the @tmp array
          $i = 2; $j = 2;
          while ($i < @alarm && $j < @ascent)
                {if ($ascent[$j] < $alarm[$i])
                    {push(@tmp, ("ASCENT", @ascent[$j .. $j+1]));
                     $j+=3;
                    }
                 elsif ($ascent[$j] > $alarm[$i])
                    {push(@tmp, (@alarm[$i-1 .. $i+1]));
                     $i+=3;
                    }
                 else     # skip @ascent alarm, @alarm wins!
                    {push(@tmp, (@alarm[$i-1 .. $i+1]));
                     $j+=3; $i+=3;
                    }
                }
          # Add the rest of @alarm or @ascent to @tmp
          if ($i < @alarm)
             {push(@tmp, (@alarm[$i-1 .. $#alarm]));}
          if ($j < @ascent)
             {my $k = $#tmp +1 ;
              push(@tmp, (@ascent[$j-1 .. $#ascent]));
              # change (negative) speeds to "ASCENT" in @tmp array
              while ($k < @tmp)
                {$tmp[$k] = "ASCENT"; $k+=3;}
             }
          # Overwrite @alarm array with @tmp:
          @tmp[0] = $#tmp / 3;;
          @alarm = @tmp;
          #
          # Now, round depths in @dprofile to maximum 2 decimal digits
          # (see $prec).
          $i = 1;
          while ($i < @dprofile)
                {$dprofile[$i] = sprintf("$prec", $dprofile[$i]); $i+=1;}
          #
          # In CCRmode, let's record the @GASswitch chronologically (this is
          # requested by logbook.cls only when bailout events occur, but…).
          if ($CCRmode == 1 && $SPevent == 0)
             {my @tmp=@GASswitch;
              my @tics=();
              my $i=2;
              for (my $k=0 ; $k < $tmp[0] ; $k++)
                  {$tics[$k]=$tmp[$i];
                   $i+=3;
                  }
              @tics = sort {$a <=> $b} @tics;
              $i=2;
              @GASswitch=($tmp[0]);
              foreach (@tics)
                  {while ($_ != $tmp[$i]) {$i+=3;}
                   push (@GASswitch, @tmp[$i-1 .. $i+1]);
                   $tmp[$i]=-1;      # erase in case of equality
                   $i=2;              # scan @GASswitch from the beginning
                  }
             }
          # NOW, we can strip last values of @alarm according to @dprofile
          if ($deltamode == 1)
             {$i = $#alarm;
              while ($alarm[$i-1] > $#dprofile)
                {my $j = pop(@alarm);
                 $j = pop(@alarm);
                 $j = pop(@alarm);
                 $i -=3;
                }
             }
          # Printing
          print("\\DeltaT{$delta} % seconds\n");
          if ($deltamode == 0)
             {print("\\TProfile{@tprofile}\n");}
          print("\\Profile{@dprofile}\n");
          print("\\GASswitches{@GASswitch}\n");
          print("\\GASlist{@GASlist}\n");
          print("\\SPlist{@SPlist}\n") if ($CCRmode == 1);
          print("\\Alarms{@alarm}\n");
          print("%\\Ascent{@ascent}\n");
          printf("\\VmaxAscent{%.1f}\n", $maxascent) if ($maxascent > 0);
          if ($#ceiling > 4)
             {$ceiling[0] = $#ceiling/2; # number of points (2 coordinates)
              print("\\Ceiling{@ceiling}\n");
             }
          # Print GF values to .tex file only if their maximum is >= $GFmin
          # (not supported by subsurface)
          if ($#gfprofile > 1)
             {my $i=0;
              $gfprofile[0]=0;
              foreach (@gfprofile)
                 {$i=$_ if ($_ > $i)}
              $gfprofile[0]=$#gfprofile;
              print("\\GFprofile{@gfprofile}\n") if ($i >= $GFmin);
             }
          #
          $lasttic=$#dprofile;
          $lasttic=$tprofile[$#tprofile] if ($deltamode == 0);
          print("\n\\LastTic{$lasttic}\n");
          my $var=int($lasttic * $delta / 60);
          print("\\Duration{$var}\n");
          # store $timeend to compute surfaceinterval for the next dive
          $timeend = $dvtime + $var;
          printf("\\TimeEnd{%02s}{%02s}\n", int($timeend/60), $timeend%60);
          # Print water temperatures: minimum (always available),
          printf("\\WaterTempMin{%.1f}\n", $watertempmin);
          # maximum and average if enough data are available to compute them
          if ($indexw > 5 && $watertempmax > $watertempmin)
             {$watertempavg = $watertempavg / $#tprofile;
              printf("\\WaterTempAvg{%.1f}\n", $watertempavg);
              printf("\\WaterTempMax{%.1f}\n", $watertempmax);
             }
          # Compute MaxDepth from the depth profile...
          $i=0;
          foreach (@dprofile)
             {$i=$_ if ($_ > $i)}
              # compare with $maxdepth from XML file (if any), and print it.
              if ($i >= $maxdepth)
                 {printf("\\MaxDepth{%.1f}\n", $i);}
              else
                 {printf("\\MaxDepth{%.1f}",$maxdepth);
                  if ($maxdepth - $i >= 0.1)
                     {printf("%%ProfileMaxDepth=%.1f\n",$i);}
                  else
                     {print"\n";}
             }
          # MaxTTS
          if ($maxTTS > 0)
             {printf("\\MaxTTS{%u}  ", ceil($maxTTS/60));
              printf("\\MaxTTSdepth{%.2f}  ", $maxTTSdepth);
              printf("\\MaxTTStime {%.2f}\n", $maxTTStic*$delta/60);
             }
         } # end of else {$#dprofile == 1}

       # Stats for each gas (in OC mode only) : compute <factor>, where
       # <factor> is <duration in minutes> * <average pressure in bars>.
       # <factor> will help to compute the SAC for Gas<n> (in l/mn):
       # SAC(Gas<n>) = TankVol * #bars  / <factor>
       if ($CCRmode == 0 && $#dprofile > 1)
          {my $i=1;
           my $sumdepth=0; my $sum=0; my $diff = 0;
           while ($i < $#GASswitch)
              {my $otu=0;
               my $mixname=$GASswitch[$i];
               # For OTU computation, first compute O2 percentage in the mix:
               my $o2 = O2_from_mixname ($mixname);
               $i+=3; my $k=0; my $k0=0; my $k1=0;
               if ($deltamode == 1)
                  {$k0 = $GASswitch[$i-2];}
               else
                  {$k0 = 0; $k0+=1 until ($tprofile[$k0] >= $GASswitch[$i-2]);}
               if ($i < $#GASswitch)
                  {if($deltamode == 1)
                     {$k1 = $GASswitch[$i+1];}
                   else
                     {$k1 = $k0; $k1+=1 until ($tprofile[$k1] >= $GASswitch[$i+1]);}
                  }
               else
                  {$k1 = $#dprofile;}
               if ($deltamode == 1)
                  {$sum=0;
                   for ($k = $k0; $k < $k1; $k++)
                       {$sum += $dprofile[$k];
                        # Erik Baker's formula: otu = time * (ppo2/500-1)^(5/6)
                        # with time in mn, ppo2 in mbar
                        $ppo2 = ($dprofile[$k]*$mbarpermeter+$airpressure) * $o2 / 100;
                        my $K=5/6;
                        if ($ppo2 > 500)
                           {$otu += ($ppo2/500-1)**$K;}
                        # Compute CNS clock
                        $cns += cns_clock($ppo2/1000) * $delta / 60;
                       }
                   # $sum = sum of the depths while using this gas
                   $sumdepth += $sum;
                   $OTUday   += $otu * $delta / 60 ;
                   # Correction for factor (trapeze instead of rectangles):
                   $sum += ($dprofile[$k1] - $dprofile[$GASswitch[$i-2]])/2;
                   # Compute avergage pressure (in bar) for this gas:
                   $sum = $sum * $mbarpermeter / 1000 + $k1 - $GASswitch[$i-2];
                   # Compute <factor> (bar*min) for this gas and store it in $sum
                   $sum = $sum * $delta / 60;
                  }
               else
                  {# if $deltamode == 0, CNS and OTU are not recorded,
                   # only AvgDepth and <factor> are computed.
                   $sum = $diff;
                   for ($k = $k0; $k < $k1; $k++)
                       {$sum+=($tprofile[$k+1]-$tprofile[$k])*($dprofile[$k+1]+$dprofile[$k])/2;
                       }
                   if ($i < $#GASswitch)
                      {$diff = ($tprofile[$k1]-$GASswitch[$i+1])*($dprofile[$k1]+$GASswitch[$i+2])/2;
                       $sum-=$diff;
                      }
                   $sumdepth += $sum;
                   # Compute average pressure (in bar) for this gas:
                   my $t1 = $lasttic;
                   $t1 = $GASswitch[$i+1] if ($i < $#GASswitch);
                   $sum = $sum * $mbarpermeter / 1000 + $t1 - $GASswitch[$i-2];
                   # Compute <factor> (bar*min) for this gas and store it in $sum
                   $sum = $sum / 60;
                  }
               #
               # Write $sum to the @mix array: search for the *first* gas
               # with a name matching $mixname (Gas3 may get index 1 or 2).
               # When the same gas is used more than once, $sum is
               # added to the previous value in $mix[$k+3].
               $k = 0;
               while($mix[$k] ne $mixname)
                     {$k=$k+4;}
               $mix[$k+3] += $sum;
               # mix[$k+3] now holds <factor> for the current gas.
               # Initialize $sum for the next run (if $deltamode == 0)
               $sum = ($tprofile[$k+1]-$GASswitch[$i+2])*($GASswitch[$i+2]+$dprofile[$k+1])/2  if ($deltamode == 0 && $i < $#GASswitch);
              } # End of while loop
           #
           # Print average depth (computed from the depth profile).
           printf("\\AvgDepth{%.1f}\n", $sumdepth/$lasttic);
           # Print Max CNS only if available.
           if ($cnsmax > 0)
              {printf("\\MaxCNS{%.0f}", $cnsmax);      # computer value
               printf(" %% %.0f", $cns) if ($cns > 0); # my computed value…
               print("\n");
              }
           else
              {printf("\\MaxCNS{%.0f}\n", $cns) if ($cns > 0);}
           # Print daily OTU if available.
           printf("\\OTUday{%.0f}\n\n", $OTUday) if ($OTUday > 0);
       }
       #
       # Print gas statistics (without duplicates), first used, first in.
       if ($CCRmode == 0 && $#dprofile > 1)
          {$i=1;
           my $var=1;
           while ($i < $GASlist[0] * 3) # loop "$GASlist[0]" times
             {my $mixname = $GASswitch[$i];
              my $k=0;
              while ($mix[$k] ne $mixname)
                    {$k=$k+4;}
              printf("\\Gas%u\{%s}{%.2f}\n",
                     $var, strip_braces($mix[$k]), $mix[$k+3]);
              print ("\\TankVolGas$var\{$tankvol[$var-1]} ");
              print ("\\PstartGas$var\{$mix[$k+1]} ");
              print ("\\PendGas$var\{$mix[$k+2]}\n\n");
              $var+=1;
              $i+= 3;
             }
          }
       #  CCR mode now (for fixed time increments only):
       if ($deltamode == 1 && $CCRmode == 1 && $#dprofile > 1)  # CCRmode now
           {# Compute average depth from the profile and print it.
            my $sumdepth=0;
            foreach (@dprofile)
                 {$sumdepth+=$_;}
            printf("\\AvgDepth{%.1f}\n", $sumdepth / $#dprofile);
            # Compute $OTUdaily for CCR mode : read array @GASswitch
            # backwards in search of SP or bailout switches.
            my $i = $#GASswitch - 2;
            my $ntick = $#dprofile;
            while ($i > 1)
              {my $mixname = $GASswitch[$i];
               my $tick = $GASswitch[$i+1];
               my $otu=0;
               # Computing OTUs for whiles spent on a fixed SP (=PPO2)
               # is straightforward using E. Baker's formula.
               if ($mixname =~ /SP=(\d\.\d)/i)
                  {$ppo2 = $1;
                   my $K=5/6;
                   if ($ppo2 > 0.5)
                      {$otu = ($ppo2 * 2 - 1)**$K *
                              ($ntick - $tick) * $delta / 60;
                      }
                   # Compute CNS clock
                   $cns += cns_clock($ppo2) * ($ntick - $tick) * $delta / 60;
                  }
               elsif ($mixname =~ /(BO=)(.*)/i)
               # For computation of OTUs for whiles spent on bailout
               # see above (OCR mode), $ppo2 has to be computed from $depth.
               # First compute O2 percentage in the mix:
                  {my $o2 = O2_from_mixname ($mixname);
                   # Loop for depths in bailout mode
                   for (my $k = $tick; $k < $ntick; $k++)
                     {$ppo2 = ($dprofile[$k]*$mbarpermeter+$airpressure) * $o2 / 100;
                      if ($ppo2 > 500)
                         {$otu += ($ppo2/500-1)**1.2;}
                      # Compute CNS clock
                      $cns += cns_clock($ppo2/1000) * $delta / 60;
                     }
                   $otu = $otu * $delta / 60;
                  }
               $OTUday += $otu;
               $ntick = $tick;
               $i -= 3;
              }
            # Print Max CNS only if available.
            if ($cnsmax > 0)
               {printf("\\MaxCNS{%.0f}", $cnsmax);      # computer value
                printf(" %% %.0f", $cns) if ($cns > 0); # my computed value…
                print("\n");
               }
            else
               {printf("\\MaxCNS{%.0f}\n", $cns) if ($cns > 0);}
            # Print daily OTU
            printf("\\OTUday{%.0f}\n\n", $OTUday);
            # Print all diluent names used with statistics
            my $var=0;
            # Check @GASswitch for diluents
            for (my $i=0 ; $i < $GASswitch[0] ; $i++)
              {unless ($GASswitch[3*$i+1] =~ /=/)
                  {my $diluent = $GASswitch[3*$i+1];
                   my $k=0;
                   while ($mix[$k] ne $diluent)
                         {$k=$k+4;}
                   $var+=1;
                   printf("\\Diluent[%u]\{%s}\n", $var,strip_braces($diluent));
                   print("\\TankVolDiluent[$var]\{$tankvol[$k/4]} ");
                   print("\\PstartDiluent[$var]\{$mix[$k+1]} ");
                   print("\\PendDiluent[$var]\{$mix[$k+2]}\n");
                  }
               }
            print("\n\\MainDiluentNR\{1}\n") if ($var > 1);
            # Is Oxygen tank volume stored anywhere?
            #print("\n\\TankVolOxygen\{$tankvol[1]}   ");
            print("\n\\TankVolOxygen\{}   ");
            print("\\PstartOxygen\{$mix[5]} ");
            print("\\PendOxygen\{$mix[6]}\n\n");
            # Check @GASswitch for bailouts
            $var=0;
            for (my $i=0 ; $i < $GASswitch[0] ; $i++)
              {if ($GASswitch[3*$i+1] =~ /BO=/)
                  {my $bailout = $GASswitch[3*$i+1];
                   $var+=1;
                   printf("\\Bailout[%u]\{%s}\n", $var,strip_braces($bailout));
                   print("\\TankVolBailout[$var]\{} ");
                   print("\\PstartBailout[$var]\{} ");
                   print("\\PendBailout[$var]\{}\n\n");
                  }
              }
           }
       #
       print("%\\UserTag{<Time in min>}{<Depth in m>}{Tag}\n\n");
       print("\\PrintData\n\n");
       print("\\begin{notes}\n$notes\\end{notes}\n\n");
       print("\\begin{private}\n\\end{private}\n\n");
       print("\\end{dive}\n");
      }
    }  # closing if ($diveNR > $LastDive) (new dives only)
 }     # end of while(<>) loop

# Add information about dive numbers at the end of the .tex file.
if ($diveNR > 0)
   {printf("%% %u dives, FirstDive=%u LastDive=%u",
                 $dives, $diveNR-$dives+1, $diveNR);
   }
else
   {print"\n% No new dives found!";}

printf("\n\n\\end{document}\n\n");

# Postambule: read from file "logbook.post" (i.e. emacs local variables)
if (open(my $in, "<", "logbook.post"))
 {while (<$in>) {print("$_");}
  close $in;
 }

# Store latest $diveNR value (last dive number) in $stampFILE, unless
# either no new dives were found, or $LastDive was given on the command line.
if ($diveNR > 0 && $UpdateStamp == 1)
   {open(STAMP, ">", "$stampDIR\/$stampFILE") ||
                     die("Can't write to $stampFILE: $!");
    print(STAMP "LastDive=$diveNR\n");
    close(STAMP);
   }

### Local Variables:
### mode: perl
### End:
